<?php
/**
 * Plugin Name: Auto License Generator
 * Plugin URI: 
 * Description: این پلاگین به‌صورت خودکار برای محصولات انتخاب شده کد لایسنس تولید می‌کند، امکان تنظیم تعداد استفاده (مثلاً یکبار مصرف) را فراهم می‌کند، صفحه‌ای برای محافظت از محتوا با وارد کردن کد لایسنس ایجاد می‌کند، یک سیستم لاگ مدیریتی جهت مدیریت کدها دارد و امکان مدیریت کدهای لایسنس در بخش سفارشات را فراهم می‌کند. همچنین در صورت تغییر وضعیت سفارش (لغو یا استرداد) کدهای لایسنس از دیتابیس حذف می‌شوند، امکان تغییر کد لایسنس توسط مدیر سایت وجود دارد و در ایمیل ارسالی به مشتری، کد لایسنس نمایش داده می‌شود.
 * Version: 1.8 Modified
 * Author: مصطفی نورمحمدی 
 * License: GPL2
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit; // جلوگیری از دسترسی مستقیم
}

class Auto_License_Generator {

    private $table_name;
    private $log_table_name;

    public function __construct() {
        global $wpdb;
        $this->table_name     = $wpdb->prefix . 'license_codes';
        $this->log_table_name = $wpdb->prefix . 'license_usage_log';

        // شروع سشن در صورت نیاز (برای ذخیره وضعیت تأیید لایسنس در صفحات محافظت‌شده)
        if ( session_id() == '' ) {
            session_start();
        }
        
        // هوک‌های مورد نیاز پلاگین
        register_activation_hook(__FILE__, array($this, 'activate_plugin'));
        add_action('add_meta_boxes', array($this, 'add_license_meta_box'));
        add_action('save_post_product', array($this, 'save_license_meta_box'), 10, 2);
        add_action('woocommerce_order_status_completed', array($this, 'process_order'), 10, 1);
        
        // حذف کد لایسنس در صورت لغو یا استرداد سفارش
        add_action('woocommerce_order_status_cancelled', array($this, 'delete_order_license_codes'));
        add_action('woocommerce_order_status_refunded', array($this, 'delete_order_license_codes'));

        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('woocommerce_order_item_meta_end', array($this, 'display_license_code_in_order'), 10, 3);

        // افزودن متاباکس در صفحه سفارش (پنل مدیریت)
        add_action('add_meta_boxes', array($this, 'add_order_license_meta_box'));

        // اکشن تغییر کد لایسنس از طریق دکمه در صفحه سفارش
        add_action('admin_post_change_license_code', array($this, 'handle_change_license_code'));

        // اکشن حذف کد لایسنس از سیستم لاگ
        add_action('admin_post_delete_license_code', array($this, 'handle_delete_license_code'));

        // فیلتر برای نمایش فرم ورود کد لایسنس در صفحه محافظت‌شده
        add_filter('the_content', array($this, 'filter_protected_content'));

        // نمایش کد لایسنس در ایمیل‌های سفارش به مشتری
        add_action('woocommerce_email_after_order_table', array($this, 'add_license_code_to_email'), 10, 4);
    }

    /**
     * ایجاد یا به‌روزرسانی جدول‌های اختصاصی در دیتابیس هنگام فعالسازی پلاگین
     */
    public function activate_plugin() {
        global $wpdb;
        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE {$this->table_name} (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            order_id bigint(20) NOT NULL,
            user_id bigint(20) NOT NULL DEFAULT 0,
            product_id bigint(20) NOT NULL,
            license_code varchar(255) NOT NULL,
            usage_limit int(11) NOT NULL,
            usage_count int(11) NOT NULL DEFAULT 0,
            disabled tinyint(1) NOT NULL DEFAULT 0,
            created_at datetime DEFAULT CURRENT_TIMESTAMP NOT NULL,
            PRIMARY KEY  (id)
        ) $charset_collate;";

        // ایجاد جدول لاگ استفاده از لایسنس
        $sql_log = "CREATE TABLE {$this->log_table_name} (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            license_id mediumint(9) NOT NULL,
            user_id bigint(20) NOT NULL DEFAULT 0,
            used_at datetime DEFAULT CURRENT_TIMESTAMP NOT NULL,
            PRIMARY KEY  (id)
        ) $charset_collate;";

        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
        dbDelta($sql_log);

        flush_rewrite_rules();
    }

    /**
     * افزودن متاباکس تنظیمات لایسنس به صفحه ویرایش محصولات
     */
    public function add_license_meta_box() {
        add_meta_box(
            'license_options',
            'تنظیمات لایسنس',
            array($this, 'render_license_meta_box'),
            'product',
            'side',
            'default'
        );
    }

    /**
     * نمایش فرم متاباکس در ویرایش محصول
     */
    public function render_license_meta_box($post) {
        $enable_license = get_post_meta($post->ID, '_enable_license', true);
        $usage_limit    = get_post_meta($post->ID, '_license_usage_limit', true);
        if(empty($usage_limit)) {
            $usage_limit = 1; // پیشفرض یکبار مصرف
        }
        ?>
        <p>
            <label for="enable_license">
                <input type="checkbox" name="enable_license" id="enable_license" value="yes" <?php checked($enable_license, 'yes'); ?> />
                فعالسازی تولید لایسنس برای این محصول
            </label>
        </p>
        <p>
            <label for="license_usage_limit">تعداد دفعات استفاده:</label>
            <input type="number" name="license_usage_limit" id="license_usage_limit" value="<?php echo esc_attr($usage_limit); ?>" min="1" />
        </p>
        <?php
    }

    /**
     * ذخیره اطلاعات متاباکس در متاهای محصول
     */
    public function save_license_meta_box($post_id, $post) {
        if (isset($_POST['enable_license']) && $_POST['enable_license'] == 'yes') {
            update_post_meta($post_id, '_enable_license', 'yes');
        } else {
            update_post_meta($post_id, '_enable_license', 'no');
        }

        if (isset($_POST['license_usage_limit'])) {
            $limit = intval($_POST['license_usage_limit']);
            if($limit < 1) {
                $limit = 1;
            }
            update_post_meta($post_id, '_license_usage_limit', $limit);
        }
    }

    /**
     * پردازش سفارش پس از تکمیل، تولید کد لایسنس برای محصولات فعال‌شده
     */
    public function process_order($order_id) {
        if(!$order_id) return;

        $order = wc_get_order($order_id);
        if(!$order) return;

        global $wpdb;

        foreach($order->get_items() as $item_id => $item) {
            $product_id   = $item->get_product_id();
            $enable_license = get_post_meta($product_id, '_enable_license', true);
            if($enable_license !== 'yes') continue;

            $usage_limit = get_post_meta($product_id, '_license_usage_limit', true);
            if(empty($usage_limit) || $usage_limit < 1) {
                $usage_limit = 1;
            }

            $quantity = $item->get_quantity();
            $user_id  = $order->get_user_id();

            for($i = 0; $i < $quantity; $i++){
                $license_code = $this->generate_license_code();
                $wpdb->insert(
                    $this->table_name,
                    array(
                        'order_id'     => $order_id,
                        'user_id'      => $user_id,
                        'product_id'   => $product_id,
                        'license_code' => $license_code,
                        'usage_limit'  => $usage_limit,
                        'usage_count'  => 0,
                        'disabled'     => 0,
                        'created_at'   => current_time('mysql')
                    ),
                    array(
                        '%d', '%d', '%d', '%s', '%d', '%d', '%d', '%s'
                    )
                );
                $license_id = $wpdb->insert_id;

                // ذخیره کد و شناسه لایسنس در متادیتای آیتم سفارش جهت استفاده‌های بعدی
                wc_add_order_item_meta( $item_id, '_license_code', $license_code );
                wc_add_order_item_meta( $item_id, '_license_id', $license_id );
            }
        }
    }

    /**
     * تولید کد لایسنس به صورت رشته‌ای تصادفی
     */
    private function generate_license_code($length = 16) {
        $characters     = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789';
        $charactersLength = strlen($characters);
        $randomString   = '';
        for($i = 0; $i < $length; $i++){
            $randomString .= $characters[rand(0, $charactersLength - 1)];
        }
        return $randomString;
    }

    /**
     * افزودن منوهای مدیریتی به داشبورد وردپرس
     */
    public function add_admin_menu() {
        add_submenu_page(
            'woocommerce',
            'تنظیمات تولید لایسنس',
            'تولید لایسنس',
            'manage_options',
            'license-generator-settings',
            array($this, 'render_settings_page')
        );
    }

    /**
     * نمایش صفحه تنظیمات پلاگین در بخش مدیریت
     */
    public function render_settings_page() {
        ?>
        <div class="wrap">
            <h1>تنظیمات تولید لایسنس</h1>
            <p>این پلاگین به‌صورت خودکار برای محصولات فعال‌شده کد لایسنس تولید می‌کند.</p>
            
            <h2>تنظیمات صفحه محافظت شده</h2>
            <?php
            if(isset($_POST['alg_settings_submit'])) {
                if(check_admin_referer('alg_update_settings', 'alg_settings_nonce')) {
                    update_option('alg_protected_page_url', sanitize_text_field($_POST['alg_protected_page_url']));
                    update_option('alg_error_invalid_license', sanitize_text_field($_POST['alg_error_invalid_license']));
                    update_option('alg_error_used_license', sanitize_text_field($_POST['alg_error_used_license']));
                    echo '<div class="updated"><p>تنظیمات به‌روز شد.</p></div>';
                }
            }
            ?>
            <form method="post">
                <?php wp_nonce_field('alg_update_settings', 'alg_settings_nonce'); ?>
                <table class="form-table">
                    <tr>
                        <th scope="row"><label for="alg_protected_page_url">آدرس صفحه محافظت شده</label></th>
                        <td>
                            <input type="text" name="alg_protected_page_url" id="alg_protected_page_url" value="<?php echo esc_attr(get_option('alg_protected_page_url', '')); ?>" class="regular-text">
                            <p class="description">مثلاً آدرس صفحه‌ای که می‌خواهید محتوا تا وارد کردن کد لایسنس نمایش داده نشود.</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="alg_error_invalid_license">پیام خطا در صورت کد نامعتبر</label></th>
                        <td>
                            <input type="text" name="alg_error_invalid_license" id="alg_error_invalid_license" value="<?php echo esc_attr(get_option('alg_error_invalid_license', 'کد لایسنس نامعتبر است.')); ?>" class="regular-text">
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="alg_error_used_license">پیام خطا در صورت استفاده‌شده کد</label></th>
                        <td>
                            <input type="text" name="alg_error_used_license" id="alg_error_used_license" value="<?php echo esc_attr(get_option('alg_error_used_license', 'کد لایسنس استفاده شده است.')); ?>" class="regular-text">
                        </td>
                    </tr>
                </table>
                <?php submit_button('ذخیره تنظیمات', 'primary', 'alg_settings_submit'); ?>
            </form>
            
            <h2>کدهای لایسنس تولید شده (سیستم لاگ)</h2>
            <?php $this->list_license_codes(); ?>
        </div>
        <?php
    }

    /**
     * نمایش لیستی از کدهای لایسنس تولید شده (۵۰ مورد آخر) در صفحه تنظیمات
     * اضافه شده: ستون عملیات با دکمه حذف
     */
    public function list_license_codes() {
        global $wpdb;
        $licenses = $wpdb->get_results("SELECT * FROM {$this->table_name} ORDER BY created_at DESC LIMIT 50");
        if($licenses) {
            echo '<table class="wp-list-table widefat fixed striped">';
            echo '<thead><tr>';
            echo '<th>ID</th><th>شماره سفارش</th><th>محصول</th><th>کد لایسنس</th><th>تعداد استفاده مجاز</th><th>تعداد استفاده شده</th><th>تاریخ ایجاد</th><th>وضعیت</th><th>عملیات</th>';
            echo '</tr></thead>';
            echo '<tbody>';
            foreach($licenses as $license) {
                echo '<tr>';
                echo '<td>' . esc_html($license->id) . '</td>';
                echo '<td>' . esc_html($license->order_id) . '</td>';
                echo '<td>' . esc_html($license->product_id) . '</td>';
                echo '<td>' . esc_html($license->license_code) . '</td>';
                echo '<td>' . esc_html($license->usage_limit) . '</td>';
                echo '<td>' . esc_html($license->usage_count) . '</td>';
                echo '<td>' . esc_html($license->created_at) . '</td>';
                echo '<td>' . ($license->disabled ? 'غیرفعال' : 'فعال') . '</td>';
                // دکمه تغییر کد لایسنس
                $change_url = admin_url('admin-post.php?action=change_license_code&license_id=' . $license->id . '&order_id=' . $license->order_id . '&_wpnonce=' . wp_create_nonce('change_license_' . $license->id));
                // دکمه حذف کد لایسنس
                $delete_url = admin_url('admin-post.php?action=delete_license_code&license_id=' . $license->id . '&_wpnonce=' . wp_create_nonce('delete_license_' . $license->id));
                echo '<td>';
                echo '<a href="' . esc_url($change_url) . '" class="button">تغییر</a> ';
                echo '<a href="' . esc_url($delete_url) . '" class="button" onclick="return confirm(\'آیا از حذف این کد لایسنس اطمینان دارید؟\');">حذف</a>';
                echo '</td>';
                echo '</tr>';
            }
            echo '</tbody>';
            echo '</table>';
        } else {
            echo 'هنوز کدی تولید نشده است.';
        }
    }

    /**
     * فیلتر محتوا برای صفحه محافظت شده.
     * در این بخش با استفاده از الگوی Post/Redirect/Get پس از تأیید موفق کد، صفحه به صورت GET نمایش داده می‌شود.
     */
    public function filter_protected_content($content) {
        if ( ! is_page() ) {
            return $content;
        }
        global $post;
        $protected_page_url = trim(get_option('alg_protected_page_url', ''));
        $current_page_url   = get_permalink($post->ID);

        if ( $protected_page_url !== '' && $protected_page_url === $current_page_url ) {

            if ( ! is_user_logged_in() ) {
                $login_form = wp_login_form( array('echo' => false) );
                return '<div class="alg-license-form"><p style="color:red;">لطفا قبل از ثبت کد در سایت وارد شوید.</p>' . $login_form . '</div>';
            }

            $user_id = get_current_user_id();
            $license_valid = false;
            $license_submitted = false;
            $error = '';

            // بررسی وجود شناسه لایسنس تأیید شده در متای کاربر
            $verified_license_id = get_user_meta( $user_id, 'alg_license_verified', true );
            if ( $verified_license_id ) {
                global $wpdb;
                $license = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$this->table_name} WHERE id = %d", $verified_license_id ) );
                if ( $license && intval($license->usage_count) < intval($license->usage_limit) && !$license->disabled ) {
                    $license_valid = true;
                } else {
                    delete_user_meta($user_id, 'alg_license_verified');
                }
            }
            
            // بررسی ارسال فرم
            if ( isset($_POST['alg_license_submit']) ) {
                $license_submitted = true;
                if ( ! isset($_POST['alg_license_nonce']) || ! wp_verify_nonce($_POST['alg_license_nonce'], 'alg_license_verify') ) {
                    $error = 'خطا در اعتبارسنجی فرم.';
                } else {
                    $submitted_code = sanitize_text_field($_POST['alg_license_code']);
                    global $wpdb;
                    $license = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$this->table_name} WHERE license_code = %s", $submitted_code ) );
                    if ( ! $license ) {
                        $error = get_option('alg_error_invalid_license', 'کد لایسنس نامعتبر است.');
                    } else {
                        if ( intval($license->usage_count) >= intval($license->usage_limit) ) {
                            $error = get_option('alg_error_used_license', 'کد لایسنس استفاده شده است.');
                        } else {
                            // افزایش تعداد استفاده کد
                            $wpdb->update( 
                                $this->table_name, 
                                array( 'usage_count' => intval($license->usage_count) + 1 ), 
                                array( 'id' => $license->id ), 
                                array( '%d' ), 
                                array( '%d' ) 
                            );
                            // ثبت استفاده در جدول لاگ
                            $wpdb->insert(
                                $this->log_table_name,
                                array(
                                    'license_id' => $license->id,
                                    'user_id'    => $user_id,
                                    'used_at'    => current_time('mysql')
                                ),
                                array('%d', '%d', '%s')
                            );
                            // ذخیره شناسه لایسنس تأیید شده در متای کاربر
                            update_user_meta($user_id, 'alg_license_verified', $license->id);
                            $license_valid = true;
                            
                            // استفاده از الگوی PRG: پس از تأیید موفق، به همان صفحه ریدایرکت می‌کنیم
                            wp_redirect( get_permalink($post->ID) );
                            exit;
                        }
                    }
                }
            }
            
            if ( $license_valid ) {
                // در درخواست GET بعد از ریدایرکت، اگر کد تأیید شده باشد، محتوای اصلی صفحه نمایش داده می‌شود.
                return $content;
            } else {
                // نمایش فرم وارد کردن کد لایسنس (با نمایش خطا در صورت وجود)
                ob_start();
                ?>
                <style>
                    /* در صورت نیاز آدرس فونت iranyekanx را اصلاح کنید */
                    @import url("https://yourdomain.com/path/to/iranyekanx.css");
                    #license-protection {
                        max-width: 500px;
                        margin: 50px auto;
                        padding: 20px;
                        border: 1px solid #ddd;
                        border-radius: 10px;
                        box-shadow: 0 2px 10px rgba(0,0,0,0.1);
                        background-color: #fff;
                        font-family: "iranyekanx", sans-serif;
                    }
                    #license-protection h2 {
                        text-align: center;
                        margin-bottom: 20px;
                        font-size: 24px;
                        color: #333;
                    }
                    #license-protection form {
                        display: flex;
                        flex-direction: column;
                    }
                    #license-protection label {
                        margin-bottom: 10px;
                        font-size: 16px;
                        color: #555;
                    }
                    #license-protection input[type="text"],
                    #license-protection input[type="submit"] {
                        padding: 10px;
                        border: 1px solid #ccc;
                        border-radius: 5px;
                        margin-bottom: 15px;
                        font-size: 16px;
                    }
                    #license-protection input[type="submit"] {
                        background-color: #0073aa;
                        color: #fff;
                        border: none;
                        cursor: pointer;
                        transition: background-color 0.3s ease;
                    }
                    #license-protection input[type="submit"]:hover {
                        background-color: #006799;
                    }
                </style>
                <div id="license-protection">
                    <h2>ورود کد لایسنس</h2>
                    <form method="post">
                        <?php wp_nonce_field('alg_license_verify', 'alg_license_nonce'); ?>
                        <p>
                            <label for="alg_license_code">کد لایسنس خود را وارد کنید:</label>
                            <input type="text" name="alg_license_code" id="alg_license_code" required>
                        </p>
                        <p>
                            <input type="submit" name="alg_license_submit" value="بررسی کد لایسنس">
                        </p>
                        <?php if($error !== ''): ?>
                            <p style="color:red;"><?php echo esc_html($error); ?></p>
                        <?php endif; ?>
                    </form>
                </div>
                <?php
                return ob_get_clean();
            }
        }
        return $content;
    }

    /**
     * نمایش کد لایسنس در جزئیات سفارش (در فرانت‌اند)
     */
    public function display_license_code_in_order( $item_id, $item, $order ) {
        $license_code = wc_get_order_item_meta( $item_id, '_license_code', true );
        if ( ! empty( $license_code ) ) {
            echo '<p><strong>کد لایسنس:</strong> ' . esc_html($license_code) . '</p>';
        }
    }

    /**
     * حذف کامل کدهای لایسنس، لاگ‌های مربوطه و متای کاربر در صورت لغو یا استرداد سفارش
     */
    public function delete_order_license_codes($order_id) {
        global $wpdb;
        // دریافت کدهای لایسنس مربوط به سفارش
        $licenses = $wpdb->get_results($wpdb->prepare("SELECT * FROM {$this->table_name} WHERE order_id = %d", $order_id));
        
        if ($licenses) {
            foreach ($licenses as $license) {
                // حذف رکوردهای لاگ مرتبط با این لایسنس
                $wpdb->delete($this->log_table_name, array('license_id' => $license->id), array('%d'));
                
                // حذف متای کاربر در صورت ثبت شناسه لایسنس
                $user_id = $license->user_id;
                $verified_license_id = get_user_meta($user_id, 'alg_license_verified', true);
                if ($verified_license_id == $license->id) {
                    delete_user_meta($user_id, 'alg_license_verified');
                }
            }
        }
        
        // حذف کدهای لایسنس از جدول اصلی
        $wpdb->delete($this->table_name, array('order_id' => $order_id), array('%d'));
    }

    /**
     * افزودن متاباکس نمایش کدهای لایسنس در صفحه ویرایش سفارش (پنل مدیریت)
     */
    public function add_order_license_meta_box() {
        add_meta_box(
            'order_license_codes',
            'کدهای لایسنس این سفارش',
            array($this, 'render_order_license_meta_box'),
            'shop_order',
            'normal',
            'default'
        );
    }

    /**
     * نمایش متاباکس کدهای لایسنس در صفحه سفارش
     */
    public function render_order_license_meta_box($post) {
        $order_id = $post->ID;
        global $wpdb;
        $licenses = $wpdb->get_results($wpdb->prepare("SELECT * FROM {$this->table_name} WHERE order_id = %d", $order_id));
        if ($licenses) {
            echo '<table class="widefat fixed striped">';
            echo '<thead><tr>';
            echo '<th>محصول</th><th>کد لایسنس</th><th>تعداد استفاده</th><th>عملیات</th>';
            echo '</tr></thead>';
            echo '<tbody>';
            foreach ($licenses as $license) {
                $product = wc_get_product($license->product_id);
                $product_name = $product ? $product->get_name() : '-';
                echo '<tr>';
                echo '<td>' . esc_html($product_name) . '</td>';
                echo '<td>' . esc_html($license->license_code) . '</td>';
                echo '<td>' . esc_html($license->usage_count) . ' / ' . esc_html($license->usage_limit) . '</td>';
                echo '<td>';
                $url = admin_url('admin-post.php?action=change_license_code&license_id=' . $license->id . '&order_id=' . $order_id . '&_wpnonce=' . wp_create_nonce('change_license_' . $license->id));
                echo '<a href="' . esc_url($url) . '" class="button">تغییر کد لایسنس</a>';
                echo '</td>';
                echo '</tr>';
            }
            echo '</tbody>';
            echo '</table>';
        } else {
            echo '<p>هیچ کد لایسنس برای این سفارش یافت نشد.</p>';
        }
    }

    /**
     * اکشن مدیریت جهت تغییر کد لایسنس (اجرای دکمه «تغییر کد لایسنس»)
     */
    public function handle_change_license_code() {
        if ( ! current_user_can('manage_options') ) {
            wp_die('شما اجازه دسترسی به این بخش را ندارید.');
        }
        if ( ! isset($_GET['_wpnonce']) || ! wp_verify_nonce($_GET['_wpnonce'], 'change_license_' . $_GET['license_id']) ) {
            wp_die('اعتبارسنجی نامعتبر.');
        }
        if ( ! isset($_GET['license_id'], $_GET['order_id']) ) {
            wp_die('پارامترهای مورد نیاز ارسال نشده‌اند.');
        }
        $license_id = intval($_GET['license_id']);
        $order_id   = intval($_GET['order_id']);
        
        $new_code = $this->generate_license_code();
        
        global $wpdb;
        $updated = $wpdb->update($this->table_name, array('license_code' => $new_code), array('id' => $license_id), array('%s'), array('%d'));
        
        if ($updated !== false) {
            $order = wc_get_order($order_id);
            if ($order) {
                foreach ($order->get_items() as $item_id => $item) {
                    $item_license_id = wc_get_order_item_meta($item_id, '_license_id', true);
                    if ($item_license_id == $license_id) {
                        wc_update_order_item_meta($item_id, '_license_code', $new_code);
                    }
                }
            }
            $redirect_url = add_query_arg('message', 'license_changed', wp_get_referer());
            wp_redirect($redirect_url);
            exit;
        } else {
            wp_die('خطا در به‌روزرسانی کد لایسنس.');
        }
    }

    /**
     * اکشن مدیریت جهت حذف کد لایسنس از سیستم لاگ
     */
    public function handle_delete_license_code() {
        if ( ! current_user_can('manage_options') ) {
            wp_die('شما اجازه دسترسی به این بخش را ندارید.');
        }
        if ( ! isset($_GET['_wpnonce']) || ! wp_verify_nonce($_GET['_wpnonce'], 'delete_license_' . $_GET['license_id']) ) {
            wp_die('اعتبارسنجی نامعتبر.');
        }
        if ( ! isset($_GET['license_id']) ) {
            wp_die('پارامترهای مورد نیاز ارسال نشده‌اند.');
        }
        $license_id = intval($_GET['license_id']);
        global $wpdb;
        
        // دریافت رکورد کد لایسنس
        $license = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$this->table_name} WHERE id = %d", $license_id ) );
        if ( ! $license ) {
            wp_die('کد لایسنس یافت نشد.');
        }
        
        // حذف متای کاربر در صورت ثبت شناسه کد لایسنس
        $user_id = $license->user_id;
        $verified_license_id = get_user_meta($user_id, 'alg_license_verified', true);
        if ($verified_license_id == $license_id) {
            delete_user_meta($user_id, 'alg_license_verified');
        }
        
        // حذف رکوردهای لاگ مرتبط با این کد لایسنس
        $wpdb->delete($this->log_table_name, array('license_id' => $license_id), array('%d'));
        
        // حذف رکورد کد لایسنس از جدول اصلی
        $wpdb->delete($this->table_name, array('id' => $license_id), array('%d'));
        
        $redirect_url = add_query_arg('message', 'license_deleted', wp_get_referer());
        wp_redirect($redirect_url);
        exit;
    }

    /**
     * افزودن کدهای لایسنس به ایمیل سفارش برای مشتری
     */
    public function add_license_code_to_email( $order, $sent_to_admin, $plain_text, $email ) {
        // فقط برای ایمیل‌های مشتری نمایش داده شود
        if ( $sent_to_admin ) {
            return;
        }
        
        $has_license = false;
        $license_data = array();

        foreach ( $order->get_items() as $item_id => $item ) {
            $license_code = wc_get_order_item_meta( $item_id, '_license_code', true );
            if ( ! empty( $license_code ) ) {
                $has_license = true;
                $product_name = $item->get_name();
                $license_data[] = array(
                    'product' => $product_name,
                    'code'    => $license_code
                );
            }
        }

        if ( $has_license ) {
            if ( $plain_text ) {
                $output  = "\n" . __( "Your License Codes:", "auto-license-generator" ) . "\n";
                foreach ( $license_data as $data ) {
                    $output .= sprintf( __( '%s: %s', "auto-license-generator" ), $data['product'], $data['code'] ) . "\n";
                }
                echo $output;
            } else {
                echo '<h2>' . __( "Your License Codes", "auto-license-generator" ) . '</h2>';
                echo '<ul>';
                foreach ( $license_data as $data ) {
                    echo '<li>' . sprintf( __( '%s: %s', "auto-license-generator" ), esc_html($data['product']), esc_html($data['code']) ) . '</li>';
                }
                echo '</ul>';
            }
        }
    }
}

new Auto_License_Generator();
